<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\TourReviewRequest;
use App\Http\Resources\Api\TourReviewResource;
use App\Models\Tour;
use App\Models\TourReview;
use App\Services\Query\QueryBuilder;
use App\Traits\Response\HasApiResponse;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class TourReviewController extends Controller
{
    use HasApiResponse;

    /**
     * Display a listing of the resource.
     *
     * @return JsonResponse
     * @throws Exception
     */
    public function index(Request $request)
    {
        $queryBuilder = new QueryBuilder(new TourReview, $request);
        $tourReviews = $queryBuilder->build()->paginate();
        $collection = TourReviewResource::collection($tourReviews->getCollection());
        $tourReviews->setCollection(collect($collection));
        return $this->send($tourReviews);
    }


    public function store(TourReviewRequest $request, $tour)
    {
        try {
            // Validate that tour exists
            $tourModel = Tour::where('id', $tour)
                ->where('enabled', true)
                ->first();

            if (!$tourModel) {
                return $this->send(null, 'Tour not found or not available', 404);
            }

            $data = $request->getSanitized();

            // Inject tour_id from the URL
            $data['tour_id'] = $tour;

            $review = TourReview::create($data);

            // Load the tour relationship
            $review->load('tour');

            // Update the tour's ratings and review count
            if ($review->tour) {
                $review->tour->increment('rates', $review->rate);
                $review->tour->increment('reviews_number');
            }

            return $this->send(
                new TourReviewResource($review),
                __('messages.tour.reviews.added'),
                201
            );
        } catch (Exception $e) {
            return $this->send(null, 'Error creating review: ' . $e->getMessage(), 500);
        }
    }

}
